/*
 *  badblocktool.c
 *
 *  Copyright (C) 2009 Wouter Cappelle
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 2 as
 * published by the Free Software Foundation.
 *
 */
#define _GNU_SOURCE

#include <unistd.h>
#include <stdio.h>
#include <stdlib.h>
#include <getopt.h>
#include <errno.h>
#include <sys/types.h>
#include <string.h>
#include <fcntl.h>
#include <sys/ioctl.h>

#include <mtd/mtd-user.h>
#include <mtd/jffs2-user.h>

#define PROGRAM "badblock"
#define VERSION "0.1"


static const char *mtd_device;
static int list = 0;
static int print = 0;
static int mark = 0;
static int unmark = 0;
static int markadress = 0;
static int unmarkadress = 0;

void display_help (void)
{
    printf("Usage: " PROGRAM " [OPTION] <mtd device>\n"
            "Gets the Bad Block information.\n"
            "\n"
            "  -l, --list           Lists the bad blocks.\n"
            "  -m, --mark <adress>  Mark a bad blocks. (Be sure to unmark!!))\n"
            "  -u, --unmark <adress>Unmark a bad blocks. (Don't unmark Factory badblocks!!!) (NOT WORKING: use U-Boot instead))\n"
            "      --help           Display this help and exit.\n"
            "      --version        Output version information and exit.\n");
    exit(0);
}

void display_version (void)
{
    printf(PROGRAM " " VERSION "\n"
            "\n"
            "Copyright (C) 2009 Wouter Cappelle \n"
            "\n"
            PROGRAM " comes with NO WARRANTY\n"
            "to the extent permitted by law.\n"
            "\n"
            "You may redistribute copies of " PROGRAM "\n"
            "under the terms of the GNU General Public Licence.\n"
            "See the file `COPYING' for more information.\n");
    exit(0);
}

void process_options (int argc, char *argv[])
{
     int error = 0;
    
    for (;;) {
        int option_index = 0;
        static const char *short_options = "lpm:u:";
        static const struct option long_options[] = {
            {"help", no_argument, 0, 0},
            {"version", no_argument, 0, 0},
            {"list", no_argument, 0, 'l'},
            {"mark", required_argument, 0, 'm'},
            {"unmark", required_argument, 0, 'u'},
            {0, 0, 0, 0},
        };

        int c = getopt_long(argc, argv, short_options,
                long_options, &option_index);
        if (c == EOF) {
            break;
        }

        switch (c) {
            case 0:
                switch (option_index) {
                    case 0:
                        display_help();
                        break;
                    case 1:
                        display_version();
                        break;
                }
                break;
            case 'l':
                list = 1;
                break;
            case 'p':
                print = 1;
                break;
            case 'm':
                mark = 1;
                markadress = strtoul(optarg, NULL, 0);
                break;
            case 'u':
                unmark = 1;
                unmarkadress= strtoul(optarg, NULL, 0);
                break;
            default:
                error = 1;
                break;
        }
    }

    if ((argc - optind) != 1 || error)
    {
        printf("((argc - optind) != 1 || error %d)\n", error);
        display_help ();
        
    }
    mtd_device = argv[optind];

}


void BadBlockTable(void)
{
    mtd_info_t meminfo;
    int fd;
    int i;

    if ((fd = open(mtd_device, O_RDWR)) < 0) {
        fprintf(stderr, "%s: %s: %s\n", PROGRAM, mtd_device, strerror(errno));
        exit(1);
    }
    if (ioctl(fd, MEMGETINFO, &meminfo) != 0) {
        fprintf(stderr, "%s: %s: unable to get MTD device info\n", PROGRAM, mtd_device);
        exit(1);
    }

    for (i = 0; i < meminfo.size; i +=  meminfo.erasesize  )
    {
        loff_t offset = i;
        int ret = ioctl(fd, MEMGETBADBLOCK, &offset);
        if (ret > 0) 
        {
            printf ("Bad block found at 0x%08x\n", (int)offset );
        }
        else if (ret < 0) 
        {
            if (errno == EOPNOTSUPP) 
            {
                fprintf(stderr, "%s: %s: Bad block check not available\n", PROGRAM, mtd_device);
                exit(1);
            }
            else 
            {
                fprintf(stderr, "\n%s: %s: MTD get bad block failed: %s\n", PROGRAM, mtd_device, strerror(errno));
                exit(1);
            }
        }
        else
        {
            if(print)printf("Block 0x%08x is good. ret:%d\n", i , ret);
        }
    }
}

void MarkBadBlock (int adress)
{
    mtd_info_t meminfo;
    loff_t offset = adress;
    int fd;
    int  ret;
    
    if ((fd = open(mtd_device, O_RDWR)) < 0) 
    {
        fprintf(stderr, "%s: %s: %s\n", PROGRAM, mtd_device, strerror(errno));
        exit(1);
    }
    if (ioctl(fd, MEMGETINFO, &meminfo) != 0) 
    {
        fprintf(stderr, "%s: %s: unable to get MTD device info\n", PROGRAM, mtd_device);
        exit(1);
    }
    if( ((meminfo.erasesize -1 ) & offset ) != 0)
    {
        fprintf(stderr, "Error: Start offset not aligned to eraseblock size! size=0x%x off=0x%x\n", meminfo.erasesize, (unsigned int)offset);
        exit(1);
    }

    printf("Setting block as bad at offset 0x%x\n", (unsigned int)offset);
    ret = ioctl(fd, MEMSETBADBLOCK, &offset);
    if (ret != 0) {
        fprintf(stderr, "MTD Marking block bad failure: %s\n", strerror(errno));
    }

    ret = ioctl(fd, MEMGETBADBLOCK, &offset);
    if (ret <= 0) {
        fprintf(stderr, "MTD Marking block is not marked bad: %s\n", strerror(errno));
    }

}

void UnMarkBadBlock (int adress)
{
    mtd_info_t meminfo;
    erase_info_t erase;
    loff_t offset = adress;
    int fd;
    int  ret;
    
    if ((fd = open(mtd_device, O_RDWR)) < 0) 
    {
        fprintf(stderr, "%s: %s: %s\n", PROGRAM, mtd_device, strerror(errno));
        exit(1);
    }
    if (ioctl(fd, MEMGETINFO, &meminfo) != 0) 
    {
        fprintf(stderr, "%s: %s: unable to get MTD device info\n", PROGRAM, mtd_device);
        exit(1);
    }
    if( ((meminfo.erasesize -1 ) & offset ) != 0)
    {
        fprintf(stderr, "Error: Start offset not aligned to eraseblock size! size=0x%x off=0x%x\n", meminfo.erasesize, (unsigned int)offset);
        exit(1);
    }
    
    ret = ioctl(fd, MEMGETBADBLOCK, &offset);
    if (ret <= 0) {
        fprintf(stderr, "MTD Marking block is not marked bad: %s\n", strerror(errno));
        exit(1);
    }

    printf("Setting block as good at offset 0x%x\n", (unsigned int)offset);
    erase.length = meminfo.erasesize;
    erase.start = offset;
    if (ioctl(fd, MEMERASE, &erase) != 0) 
    {
        fprintf(stderr, "\n%s: %s: MTD Erase failure: %s\n", PROGRAM, mtd_device, strerror(errno));
        exit(1);
    }

    ret = ioctl(fd, MEMGETBADBLOCK, &offset);
    if (ret > 0) {
        fprintf(stderr, "MTD Marking block good failure: %s\n", strerror(errno));
        exit(1);
    }
    printf("MTD Marking block good succeeded at offset 0x%x\n",(unsigned int)offset);

}

int main(int argc, char* argv[])
{
    process_options(argc, argv);
    if( list ) BadBlockTable();
    if( mark ) MarkBadBlock(markadress);
    if( unmark ) UnMarkBadBlock(unmarkadress);
    printf("Done..\n");
    return 0;
}
